<?php

declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;

class LiveStreamingWatchesTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('live_streaming_watch');
		$this->setPrimaryKey('id');

		$this->belongsTo('Users', [
			'foreignKey' => 'user_id',
			'joinType' => 'INNER',
			'className' => 'App\Model\Table\UsersTable',
		]);

		$this->belongsTo('LiveStreamings', [
			'foreignKey' => 'live_streaming_id',
			'joinType' => 'INNER',
			'className' => 'App\Model\Table\LiveStreamingsTable',
		]);
	}

	public function getDetails(int $id): array
	{
		$record = $this->find()
			->where(['id' => $id])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function countCoinsSpendByUser(int $user_id): array
	{
		$result = $this->find()
			->select(['total_amount' => $this->find()->func()->sum('coin')])
			->where(['user_id' => $user_id])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function checkDuplicate(int $user_id, int $live_streaming_id): array
	{
		$record = $this->find()
			->where([
				'user_id' => $user_id,
				'live_streaming_id' => $live_streaming_id
			])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function checkIfExist(int $user_id, int $live_streaming_id): array
	{
		$record = $this->find()
			->where([
				'user_id' => $user_id,
				'live_streaming_id' => $live_streaming_id,
				'ended_at' => '0000-00-00 00:00:00'
			])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function getLiveStreamViewers(int $live_streaming_id): array
	{
		return $this->find()
			->select(['user_id'])
			->where([
				'live_streaming_id' => $live_streaming_id,
				'duration' => 0
			])
			->group('user_id')
			->toArray();
	}

	public function getTotalDurationUserWatchedInAMonth(int $user_id, int $month): array
	{
		$result = $this->find()
			->select([
				'total_duration' => $this->find()->func()->sum('duration')
			])
			->where([
				'user_id' => $user_id,
				'MONTH(created)' => $month
			])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function checkDuplicateAll(int $user_id, int $live_streaming_id): array
	{
		return $this->find()
			->where([
				'user_id' => $user_id,
				'live_streaming_id' => $live_streaming_id
			])
			->toArray();
	}

	public function getTopViewersInAMonth(int $month): array
	{
		return $this->find()
			->select([
				'user_id',
				'total_duration' => $this->find()->func()->sum('duration')
			])
			->contain(['Users'])
			->where([
				'MONTH(created)' => $month
			])
			->group('user_id')
			->order(['total_duration' => 'DESC'])
			->limit(10)
			->toArray();
	}
}
