<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\Auth\DefaultPasswordHasher;
use Cake\Utility\Security;
use Cake\Validation\Validator;

class AdminsTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);
		$this->setTable('admin');
	}

	public function isEmailAlreadyExist(string $email): int
	{
		return $this->find()->where(['email' => $email])->count();
	}

	public function getAdminCount(): int
	{
		return $this->find()->count();
	}

	public function verifyPassword(string $email, string $old_password): bool
	{
		$userData = $this->find()->select(['id', 'password'])->where(['email' => $email])->first();
		if (empty($userData)) {
			return false;
		}

		$passwordHasher = new DefaultPasswordHasher();
		return $passwordHasher->check($old_password, $userData->password);
	}

	public function updatePassword(string $password): array
	{
		$passwordHasher = new DefaultPasswordHasher();
		return [
			'password' => $passwordHasher->hash($password),
			'salt' => Security::hash($password, 'sha256', true)
		];
	}

	public function getEmailBasedOnUserID(int $user_id): array
	{
		return $this->find()->where(['id' => $user_id])->all()->toArray();
	}

	public function verify(string $email, string $user_password): bool
	{
		if ($email !== "") {
			$userData = $this->find()->where(['email' => $email])->first();
			if (empty($userData)) {
				return false;
			}
		} else {
			return false;
		}

		$passwordHasher = new DefaultPasswordHasher();
		return $passwordHasher->check($user_password, $userData->password);
	}

	public function loginAllUsers(string $email, string $user_password): array|false
	{
		if ($email !== "") {
			$userData = $this->find()->where(['email' => $email])->first();
			if (empty($userData)) {
				return false;
			}
		} else {
			return false;
		}

		$passwordHasher = new DefaultPasswordHasher();
		if ($passwordHasher->check($user_password, $userData->password) && $userData->active == 1) {
			return $userData->toArray();
		}

		return false;
	}

	public function getUserDetailsFromID(int $user_id): array
	{
		$user = $this->find()->where(['id' => $user_id])->first();
		return $user ? $user->toArray() : [];
	}

	public function countAdminUsers(): int
	{
		return $this->find()->count();
	}

	public function loginAllUsersExceptAdmin(string $email, string $user_password, string $role): array|string|false
	{
		if ($email !== "") {
			$userData = $this->find()->where(['email' => $email, 'role' => $role])->first();
			if (empty($userData)) {
				return false;
			}
		} else {
			return false;
		}

		$passwordHasher = new DefaultPasswordHasher();
		if ($passwordHasher->check($user_password, $userData->password)) {
			if ($userData->role !== "admin") {
				return $userData->toArray();
			} else {
				return "203";
			}
		}

		return false;
	}

	public function loginRestaurantAndRiderAndUser(string $email, string $user_password): array|string|false
	{
		$usersTable = \Cake\ORM\TableRegistry::getTableLocator()->get('Users');
		$userData = $usersTable->find()->where(['email' => $email])->first();

		if (empty($userData)) {
			return false;
		}

		$passwordHasher = new DefaultPasswordHasher();
		if ($passwordHasher->check($user_password, $userData->password)) {
			if (in_array($userData->role, ['user', 'hotel', 'rider'])) {
				return $userData->toArray();
			} else {
				return "203";
			}
		}

		return false;
	}

	public function getAllUsers(): array
	{
		return $this->find()->order(['id' => 'DESC'])->all()->toArray();
	}

	public function getAdminDetails(): array
	{
		$admin = $this->find()->where(['role' => 0])->first();
		return $admin ? $admin->toArray() : [];
	}

	public function findEmail(string $email, string $role): array
	{
		$usersTable = \Cake\ORM\TableRegistry::getTableLocator()->get('Users');
		return $usersTable->find()->where(['role' => $role, 'email' => $email])->all()->toArray();
	}

	public function getAll(): array
	{
		return $this->find()->all()->toArray();
	}

	public function beforeSave(\Cake\Event\EventInterface $event, \ArrayObject $entity, \ArrayObject $options): bool
	{
		if (isset($entity['password'])) {
			$passwordHasher = new DefaultPasswordHasher();
			$password = $entity['password'];
			$entity['password'] = $passwordHasher->hash($password);
			$entity['salt'] = Security::hash($password, 'sha256', true);
		}

		return true;
	}
}
