<?php

declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;

class WithdrawRequestsTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('withdraw_request');
		$this->setPrimaryKey('id');

		$this->belongsTo('Users', [
			'foreignKey' => 'user_id',
			'className' => 'App\Model\Table\UsersTable',
		]);
	}

	public function getDetails(int $id): array
	{
		return $this->find()
			->where(['WithdrawRequest.id' => $id])
			->enableHydration(false)
			->first() ?? [];
	}

	public function getWithdrawRequestsAgainstUserID(int $user_id, int $starting_point): array
	{
		return $this->find()
			->where(['WithdrawRequest.user_id' => $user_id])
			->limit(10)
			->offset($starting_point * 10)
			->order(['WithdrawRequest.id' => 'DESC'])
			->enableHydration(false)
			->all()
			->toList();
	}

	public function getUserPendingWithdrawRequest(int $user_id): array
	{
		return $this->find()
			->where([
				'WithdrawRequest.user_id' => $user_id,
				'WithdrawRequest.status' => 0
			])
			->enableHydration(false)
			->all()
			->toList();
	}

	public function getUserLastWithdrawRequest(int $user_id): ?array
	{
		return $this->find()
			->where([
				'WithdrawRequest.user_id' => $user_id,
				'WithdrawRequest.status' => 1
			])
			->order(['WithdrawRequest.id' => 'DESC'])
			->enableHydration(false)
			->first() ?: null;
	}

	public function getTotalCoins(int $user_id): ?float
	{
		$result = $this->find()
			->select(['total_coin' => $this->find()->func()->sum('WithdrawRequest.coin')])
			->where([
				'WithdrawRequest.user_id' => $user_id,
				'WithdrawRequest.status <' => 2
			])
			->enableHydration(false)
			->first();

		return isset($result['total_coin']) ? (float)$result['total_coin'] : null;
	}

	public function getTotalAmountWithdrawal(int $user_id): ?array
	{
		$result = $this->find()
			->select(['total_amount' => $this->find()->func()->sum('WithdrawRequest.amount')])
			->where([
				'WithdrawRequest.user_id' => $user_id,
				'WithdrawRequest.status' => 1
			])
			->enableHydration(false)
			->first();

		return $result ?: null;
	}

	public function getAllPendingRequests(int $status): array
	{
		return $this->find()
			->where(['WithdrawRequest.status IN' => [$status, 1, 2]])
			->enableHydration(false)
			->all()
			->toList();
	}

	public function getAll(): array
	{
		return $this->find()
			->enableHydration(false)
			->all()
			->toList();
	}
}
